/**
 * Copyright (C) 2008 Institut fuer Telematik, Universitaet Karlsruhe (TH)
 *
 * This file is part of openser, a free SIP server.
 *
 * openser is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version
 *
 * openser is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
 * @file p2psip.c
 * @author Ingmar Baumgart
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

#include "../../dprint.h"
#include "../../action.h"
#include "../../config.h"
#include "../../ut.h"
#include "../../parser/parse_methods.h"
#include "../../parser/msg_parser.h"
#include "../../parser/parse_uri.h"
#include "../../sr_module.h"
#include "../../parser/contact/contact.h"

#include "../usrloc/usrloc.h"
#include "../registrar/sip_msg.h"
#include "../registrar/save.h"

#include "p2psip.h"
#include "xmlrpc_connect.h"

#define DEFAULT_P2PNS_PORT 3631
#define DEFAULT_P2PNS_HOST "localhost"
#define DEFAULT_P2PNS_HOST_LEN (sizeof(DEFAULT_P2PNS_HOST) - 1)

MODULE_VERSION

static int mod_init(void);
static void destroy(void);

int p2pns_port=DEFAULT_P2PNS_PORT;
str p2pns_host=str_init(DEFAULT_P2PNS_HOST);

extract_contact_and_aor_t reg_extract_contact_and_aor;

/*
 * Exported functions   (null terminated command name, pointer to the corresponding function, number of parameters used by the function,pointer to the function called to "fix" the parameters, Function flags)
 */
static cmd_export_t cmds[] = {
    {"pw_check", (cmd_function)pw_check, 0, 0, 0,
     REQUEST_ROUTE | ONREPLY_ROUTE | FAILURE_ROUTE | BRANCH_ROUTE },
    {"p2pns_register", (cmd_function)p2pns_register, 2, p2psip_fixup, 0,
     REQUEST_ROUTE | ONREPLY_ROUTE | FAILURE_ROUTE | BRANCH_ROUTE },
    {"p2pns_resolve",(cmd_function)p2pns_resolve, 1, p2psip_fixup, 0,
     REQUEST_ROUTE | ONREPLY_ROUTE | FAILURE_ROUTE | BRANCH_ROUTE },
    {"p2pns_lookup",(cmd_function)p2pns_lookup, 0, 0, 0,
          REQUEST_ROUTE | FAILURE_ROUTE },
    {"p2pns_save",(cmd_function)p2pns_save, 0, 0, 0,
          REQUEST_ROUTE },
	{0, 0, 0, 0, 0, 0}
};


/*
 * Exported parameters
 */
static param_export_t params[] = {
        {"p2pns_host", STR_PARAM, &p2pns_host.s},
        {"p2pns_port", INT_PARAM, &p2pns_port},
	{0, 0, 0}
};

/** module exports */
struct module_exports exports= {

    "p2psip",
    DEFAULT_DLFLAGS,/* dlopen flags */
    cmds,           /* Exported functions */
    params,         /* Exported parameters */
    0,              /* exported statistics */
    0,              /* exported MI functions */
    0,              /* exported pseudo-variables */
    0,              /* extra processes */
    mod_init,       /* initialization module */
    0,              /* response function */
    destroy,        /* destroy function */
    0               /* per-child init function */
};


static int mod_init(void)
{
    LM_DBG("p2psip: mod_init()\n");

	p2pns_host.len = strlen(p2pns_host.s);

	reg_extract_contact_and_aor = (extract_contact_and_aor_t)
	                              find_export("extract_contact_and_aor", 1, 0);

	if (!reg_extract_contact_and_aor) {
	    LM_ERR("can't bind registrar\n");
	    return -1;
	}

	return 0;
}

static void destroy(void)
{
    LM_DBG("p2psip: destroy()\n");
}


/**
 * Resolve AoR with P2PNS
 */
int p2pns_lookup(struct sip_msg* msg, char* par1, char* par2)
{
    #define MAX_USERURI_SIZE 256
    static char useruri_buf[MAX_USERURI_SIZE];
    static char p2pns_name_buf[MAX_USERURI_SIZE];

    if (parse_sip_msg_uri(msg) < 0)
        return -1;

    int len=0;

    strlower(&msg->parsed_uri.user);

    memcpy(p2pns_name_buf, msg->parsed_uri.user.s,
           msg->parsed_uri.user.len);
    len += msg->parsed_uri.user.len;

    p2pns_name_buf[len] = 0;

    /*
    p2pns_name_buf[len] = '@';
    len++;
    memcpy(p2pns_name_buf + len, msg->parsed_uri.host.s,
               msg->parsed_uri.host.len);
    len += msg->parsed_uri.host.len;
    */

    memset(useruri_buf, 0, MAX_USERURI_SIZE);

    if (p2pns_resolve_xmlrpc(p2pns_name_buf, useruri_buf, p2pns_host.s,
                             p2pns_port) < 0) {
        LM_DBG("failed to query P2PNS for R-URI\n");

        return -1;
    }

    if (strlen(useruri_buf) <= 0) {
        LM_DBG("P2PNS returned invalid R-URI\n");

        return -1;
    }

    /* set the URI */
    LM_DBG("the URI of alias from R-URI [%s]\n", useruri_buf);
    if (rewrite_ruri(msg, useruri_buf)<0) {
        LM_ERR("cannot replace the R-URI\n");
        return -1;
    }

    return 1;
}


/**
 * Store contact with P2PNS
 */
int p2pns_save(struct sip_msg* msg, char* par1, char* par2)
{
    contact_t* c;
    str aor;

    if (reg_extract_contact_and_aor(msg, &c, &aor) < 0) {
        LM_ERR("failed to extract Address Of Record\n");
        goto error;
    }

    if (c == 0) {
        LM_ERR("p2pns_save: no contact\n");
        goto error;
    } else {
        strlower(&aor);
        if (p2pns_register_xmlrpc(aor, c->uri, p2pns_host.s,
                              p2pns_port) < 0) goto error;
    }

    return 1;

    error:

    return 0;
}


/**
 * Rewrite Request-URI (code copied from alias_db.c)
 */
static inline int rewrite_ruri(struct sip_msg* _m, char* _s)
{
    struct action act;

    act.type = SET_URI_T;
    act.elem[0].type = STRING_ST;
    act.elem[0].u.string = _s;
    act.next = 0;

    if (do_action(&act, _m) < 0)
    {
        LM_ERR("do_action failed\n");
        return -1;
    }
    return 0;
}

int pw_check(struct sip_msg* msg, char* par1, char* par2) {
    return 1;
}


int p2pns_resolve(struct sip_msg* msg, char* par1, char* par2)
{
    #define P2PNS_BUF_SIZE 1024
    static char p2pns_name_buf[P2PNS_BUF_SIZE];
    int buf_len;

    if(msg==0 || par1==0)
        return -1;

    buf_len = P2PNS_BUF_SIZE-1;
    if(pv_printf(msg, (pv_elem_t*)par1, p2pns_name_buf, &buf_len)<0) {
        LM_ERR("cannot print the format\n");
        return -1;
    }

    return 0;
}


int p2pns_register(struct sip_msg* msg, char* par1, char* par2)
{
    #define P2PNS_BUF_SIZE   1024
    static char p2pns_name_buf[P2PNS_BUF_SIZE];
    static char p2pns_addr_buf[P2PNS_BUF_SIZE];
    str name, addr;
    int buf_len;

    if(msg==0 || par1==0 || par2==0)
       return -1;

    buf_len = P2PNS_BUF_SIZE-1;
    if(pv_printf(msg, (pv_elem_t*)par1, p2pns_name_buf, &buf_len)<0) {
        LM_ERR("cannot print the format\n");
        return -1;
    }

    buf_len = P2PNS_BUF_SIZE-1;
    if(pv_printf(msg, (pv_elem_t*)par2, p2pns_addr_buf, &buf_len)<0) {
        LM_ERR("cannot print the format\n");
        return -1;
    }

    name.s = p2pns_name_buf;
    name.len = strlen(p2pns_name_buf);
    addr.s = p2pns_addr_buf;
    addr.len = strlen(p2pns_addr_buf);
    p2pns_register_xmlrpc(name, addr, p2pns_host.s,
                          p2pns_port);

    return 0;
}


/*
 * Convert char* parameter to pv_elem parameter
 */
static int p2psip_fixup(void** param, int param_no)
{
    pv_elem_t *model;
    str s;
    if(*param)
    {
        s.s = (char*)(*param); s.len = strlen(s.s);
        if(pv_parse_format(&s, &model)<0)
        {
            LM_ERR("wrong format[%s]\n",
                (char*)(*param));
            return E_UNSPEC;
        }
        *param = (void*)model;
    }
    return 0;
}
